/*
* au - user accounting
*
* Synopsis: au [-tpgmw] [-d fromdate todate] [-r n]
*
* Options:
*   -g   Causes sorting by department (default is by user).
*   -p   Says that the output is to be formated for the printer in
*        the form of a report by department (the -g flag is auto-
*        matically supplied).
*   -t   Suppresses the generation of certain titles and informatory
*        messages.
*   -m   Provide information for most recent month (1-31st).
*   -w   Provide information for most recent week (sat-fri).
*   -d   Specifies a certain "from" date and an optional "to" date.
*        If the "to" date is omitted, only the information for the
*        "from" date will be printed.
*   -r   Must be followed by an integer number.  The accounting
*        information will be printed for the most recent "n" days.
*/

#include <stdio.h>
#include <time.h>
#include "acct.h"
#include <sys/acctfile.h>

#define  DAYSECS  (24*60*60)
#define  BYTEPWD  4
#define  MAXLINES 55

int  tflg, pflg, gflg, rflg;
int  from, to;
int  fyr,  fday, tyr,  tday;

char usage[]   = "usage: au [-gptmw] [-d from to] [-r n]";
char accttab[] = "/etc/accttab";
char dayfile[30];
char sfrom[10], sto[10];
char ss[12];

struct aday aday[MAXUSERS];

struct tots {
	char t_name[8];
	char t_dept[5];
	int  t_cmds;
	int  t_scpu;
	int  t_ucpu;
	int  t_uio;
	int  t_uconn;
	int  t_rread;
	int  t_rpun;
	int  t_rprt;
	int  t_ublk;
} tots[MAXUSERS];

int index[MAXUSERS], numb;

/*
*  rate (weight) table contains inter-relation factors for the different
*  accounting categories.
*/
struct rate {
	float  i_cmds;
	float  i_scpu;
	float  i_ucpu;
	float  i_uio;
	float  i_uconn;
	float  i_rread;
	float  i_rpun;
	float  i_rprt;
	float  i_ublk;
} rate ;
float ratesum;

/*
*  Similar structures are used to contain the per-category totals
*  and the individual user percentages.
*/
struct tots total;
struct rate perc[MAXUSERS];
float overall[MAXUSERS];


main(argc,argv) char **argv;
{
	while(--argc > 0 && *argv[1]=='-') {
		argv++;
		while(*++*argv) switch(**argv) {
		case 't':
			tflg++;
			continue;
		case 'p':
			pflg++;
			gflg++;
			continue;
		case 'g':
			gflg++;
			continue;
		case 'm':
			lastmon();
			continue;
		case 'w':
			lastweek();
			continue;
		case 'd':
			if (--argc==0)
				goto nextarg;
                        from= any(*++argv, '/')?julian(*argv):atoi(*argv);
			--argc;
			if (*argv[1]=='-')
				goto nextarg;
			to  = any(*++argv, '/')?julian(*argv):atoi(*argv);
			--argc;
			goto nextarg;
		case 'r':
			rflg = atoi(*++argv);
			--argc;
			goto nextarg;
		default:
			printf("%s\n", usage);
			exit(1);
		}
nextarg:;
	}

	if (argc > 0) {
		printf("%s\n", usage);
		exit(1);
	}

	if (rflg || (!from && !to)) maker(rflg? rflg:7);
	maked(from, to);

	getdata();
	sort();
	average();
	output();
}


/*
* lastmon: fills the from-to date values with dates ranging from the
*          first of last month until the end of last month.
*/
lastmon()
{
	int t, mon;
        struct tm *tm;
        struct tm *localtime();

	t = time();
	tm = localtime(t);

	mon = tm->tm_mon;
	while (mon == tm->tm_mon) {
		t -= DAYSECS;
		tm = localtime(t);
	}
	to = tm->tm_year*1000 + tm->tm_yday+1;

	t -= DAYSECS*25;
	while (tm->tm_mday  != 1) {
		t -= DAYSECS;
		tm = localtime(t);
	}
	from = tm->tm_year*1000 + tm->tm_yday+1;
}


/*
* lastweek: fills the from-to date values with dates ranging from the
*           first of last week (saturday) until the end (friday).
*/
lastweek()
{
	int t;
	struct tm *tm;
	struct tm *localtime();

	t = time() - DAYSECS;
	tm = localtime(t);

	while (tm->tm_wday != 5) {
		t -= DAYSECS;
		tm = localtime(t);
	}
	to = tm->tm_year*1000 + tm->tm_yday+1;

	while (tm->tm_wday != 6) {
		t -= DAYSECS;
		tm = localtime(t);
	}
	from = tm->tm_year*1000 + tm->tm_yday+1;
}


/*
* maker: fills the from-to date values with dates ranging "n" days
*        ago until yesterday.
*/
maker(n)
{
	int i, t;
	struct tm *tm;
	struct tm *localtime();

	t = time() - DAYSECS;
	tm = localtime(t);
	to = tm->tm_year*1000 + tm->tm_yday+1;

	for (i=1; i<n; i++)
		t -= DAYSECS;
	tm = localtime(t);
	from = tm->tm_year*1000 + tm->tm_yday+1;
}


/*
* maked: loads the "fyr-fday-tyr-tday" variables with the range
*        given in the julian "from-to" dates.
*/
maked(fj, tj)
{
	char *greg();

	if (!sfrom[0]) strcpy(sfrom, greg(fj));
	if (!sto[0])   strcpy(sto  , greg(tj? tj:fj));

	fyr  = fj/1000;
	fday = fj%1000;
	if (to) {
		tyr  = tj/1000;
		tday = tj%1000;
	} else {
		tyr  = fyr;
		tday = fday;
	}
}


/*
* getdata: sums all the "aday" data for the given range of days into
*          the "tots" structure.
*/
getdata()
{
	int n, max;

	while ((fyr*1000+fday) <= (tyr*1000+tday)) {
		if (getaday(fyr, fday)==0) {
			for (n=0; n<MAXUSERS; n++) {
				tots[n].t_cmds  += aday[n].nocmd;
				tots[n].t_scpu  += aday[n].scpu ;
				tots[n].t_ucpu  += aday[n].ucpu ;
				tots[n].t_uio   += aday[n].uio  ;
				tots[n].t_uconn += aday[n].uconn;
				tots[n].t_rread += aday[n].rread;
				tots[n].t_rpun  += aday[n].rpun ;
				tots[n].t_rprt  += aday[n].rprt ;
				tots[n].t_ublk  = tots[n].t_ublk==0 ?
						    aday[n].ublk :
						   (aday[n].ublk +
						    tots[n].t_ublk)/2;
			}
		}
		max = (fyr%4)==0 ? 366 : 365;
		if (++fday > max) {
			++fyr;
			fday = 1;
		}
	}
	fillrest();
}

#include <pwd.h>
struct passwd *pw;
struct passwd *getpwuid();

/*
* fillrest: fills the user name and department into the 'tots' structure.
*/
fillrest()
{
	int n;

	for (n=0; n<MAXUSERS; n++) {
		if((tots[n].t_cmds != 0  ||  tots[n].t_ublk != 0)  &&
		(pw = getpwuid(n)) != NULL ) {
			strcpy(tots[n].t_name, pw->pw_name);
			strcpy(tots[n].t_dept, pw->pw_dept);
		}
	}
}


/*
* getaday: read in one days worth of data into "aday"
*/
getaday(yr, day)
{
	int  lotime, hitime, users;
	FILE *tfd;

	sprintf(dayfile, daymodel, yr*1000 + day);

	zeroout((int *)aday, sizeof aday);
	if ((tfd = fopen(dayfile, "r"))==NULL)
		return(1);

	lotime = getw(tfd);
	hitime = getw(tfd);
	users  = getw(tfd);
	if (lotime==EOF || hitime==EOF || users==EOF) {
		fclose(tfd);
		return(1);
	}
	fread((char *)aday, sizeof *aday, users, tfd);
	fclose(tfd);
	return(0);
}


/*
* sort: sorts an "index" array pointing to the "tots" elements.
*/
sort()
{
	int n;
	extern int compar();

	for (n=0; n<MAXUSERS; n++) {
		if (tots[n].t_name[0])
			index[numb++] = n;
	}
	qsort(index, numb, sizeof *index, compar);
}


/*
* compar: compares the "tots" elements pointed to by the arguments.
*/
compar(i1, i2) int *i1, *i2;
{
	int i, j, n;

	i = *i1;
	j = *i2;
	if(gflg) {
		n = strcmp(tots[i].t_dept, tots[j].t_dept);
		if (n != 0)
			return(n);
	}
	return(strcmp(tots[i].t_name, tots[j].t_name));
}

float zero;
/*
* average: sums each category into 'total', figures the individual
*          category percent for each user (placed into 'perc') and
*          figures the percent each user used the system overall
*          into 'overall'.
*/
average()
{
	int n;
	struct tots *tp;
	float percent();

	/* do some initialization */
	makerate();
	zero = 0;

	total.t_cmds = 0;
	total.t_scpu = 0;
	total.t_ucpu = 0;
	total.t_uio  = 0;
	total.t_uconn= 0;
	total.t_rread= 0;
	total.t_rpun = 0;
	total.t_rprt = 0;
	total.t_ublk = 0;

	/* count totals */
	for (n=0; n<numb; n++) {
		tp = &tots[index[n]];
		total.t_cmds += tp->t_cmds ;
		total.t_scpu += tp->t_scpu ;
		total.t_ucpu += tp->t_ucpu ;
		total.t_uio  += tp->t_uio  ;
		total.t_uconn+= tp->t_uconn;
		total.t_rread+= tp->t_rread;
		total.t_rpun += tp->t_rpun ;
                total.t_rprt += tp->t_rprt ;
		total.t_ublk += tp->t_ublk ;
	}

	/* category percents */
	for (n=0; n<numb; n++) {
		tp = &tots[index[n]];
		perc[n].i_cmds  = percent(tp->t_cmds , total.t_cmds);
		perc[n].i_scpu  = percent(tp->t_scpu , total.t_scpu);
		perc[n].i_ucpu  = percent(tp->t_ucpu , total.t_ucpu);
		perc[n].i_uio   = percent(tp->t_uio  , total.t_uio );
		perc[n].i_uconn = percent(tp->t_uconn, total.t_uconn);
		perc[n].i_rread = percent(tp->t_rread, total.t_rread);
		perc[n].i_rpun  = percent(tp->t_rpun , total.t_rpun);
		perc[n].i_rprt  = percent(tp->t_rprt , total.t_rprt);
		perc[n].i_ublk  = percent(tp->t_ublk , total.t_ublk);
	}

	/* overall pro-rated percent for each user */
	for (n=0; n<numb; n++) {
		overall[n] = perc[n].i_cmds   * rate.i_cmds;
		overall[n] += perc[n].i_scpu  * rate.i_scpu ;
		overall[n] += perc[n].i_ucpu  * rate.i_ucpu ;
		overall[n] += perc[n].i_uio   * rate.i_uio  ;
		overall[n] += perc[n].i_uconn * rate.i_uconn;
		overall[n] += perc[n].i_rread * rate.i_rread;
		overall[n] += perc[n].i_rpun  * rate.i_rpun ;
		overall[n] += perc[n].i_rprt  * rate.i_rprt ;
		overall[n] += perc[n].i_ublk  * rate.i_ublk ;
		overall[n] = overall[n] / ratesum;
	}
}

/*
* percent: calculate  i/j * 100.0
*/
float percent(i, j) int i, j;
{
	float f;

	if (i == 0  ||  j == 0)
		return(zero);
	f = ((float)i / (float)j) * 100;
	return(f);
}


/*
* makerate: reads the rate values from 'accttab' to initialize
*           'rate' and 'ratesum'.
*/
makerate()
{
	float *fp;
	FILE *fd;
	char line[60];
	char *ctime();

	if ((fd=fopen(accttab, "r"))==NULL) {
		printf("Cannot open %s\n", accttab);
		exit(1);
	}
	if (pflg)
		printf("\n\n\nAU Accounting Weight Table   %s\n", ctime(time()));

        for(fp = &rate.i_cmds; fp <= &rate.i_ublk; fp++) {
		if (fgets(line, 60, fd)==NULL) {
			printf("Bad rate file. %s\n", accttab);
			exit(1);
		}
		sscanf(line, "%f", fp);
		if (pflg) fputs(line, stdout);
        }
	if (fgets(line, 60, fd)!=NULL) {
		printf("Bad rate file %s\n", accttab);
		exit(1);
	}
	fclose(fd);

	ratesum = rate.i_cmds + rate.i_scpu + rate.i_ucpu + rate.i_uio  +
		  rate.i_uconn+ rate.i_rread+ rate.i_rpun + rate.i_rprt +
		  rate.i_ublk ;
}


/*
* output: writes out the data ... what else?
*/
output()
{
	int n;
	char *scsec();
	struct tots *tp;

	if (pflg) {
		report();
		return;
	}
	if (!tflg) {
		if (strcmp(sfrom, sto)==0)
			printf("Accounting Summary for %s\n\n", sfrom);
		else
			printf("Accounting Summary for %s to %s\n\n", sfrom, sto);
		printf("user____dept___connect____#cmds______cpu____rd/wrt____urio__blocks__%%usage\n");
	}
	for (n=0; n<numb; n++) {
		tp = &tots[index[n]];
		if (tp->t_cmds>0) {
			printf("%-8s%-4s%10s%9d%9.02f%10d%8d%8d%8.02f\n", tp->t_name,
			tp->t_dept, scsec(tp->t_uconn, 10), tp->t_cmds,
		        (tp->t_scpu + tp->t_ucpu)/(float)1000, tp->t_uio,
			tp->t_rread + tp->t_rpun + tp->t_rprt, tp->t_ublk,
		        overall[n]);
		}
	}
}


/*
* report: generate a per-user report
*/
report()
{
	struct rate dperc;
	struct tots dtot;
	int  ifr, ito, i, page, line;
	float totperc;
	char sdate[24];

	page = 1;
	if (strcmp(sfrom,sto)==0)
		sprintf(sdate, "%s", sfrom); else
		sprintf(sdate, "%s to %s", sfrom, sto);

	for (ifr=0; ifr<numb && tots[index[ifr]].t_dept; ifr=ito+1) {
		for (ito=ifr; ito<numb && strcmp(tots[index[ito]].t_dept,
			tots[index[ifr]].t_dept)==0; ito++) ;
		ito--;

		deptcnt(&dtot, &dperc, &totperc, ifr, ito);
		strcpy(dtot.t_dept, tots[index[ifr]].t_dept);

		line = puthead(sdate, dtot.t_dept, page);
		line = puttot (&dtot,  line, -1);
		line = putperc(&dperc, totperc, line);

		printf("\n\n\n");
		for (i=ifr; i<=ito; i++) {
			if (line > MAXLINES) {
				page++;
				puthead(sdate, dtot.t_dept, page);
			}
			line = puttot(&tots[index[i]], line, i);
		}
	}
}


/*
* deptcnt: counts the totals for a department
*/
deptcnt(tp, pp, tot, ifr, ito)
struct tots *tp;
struct rate *pp;
float  *tot;
{
	int i;

	zeroout((int *)tp, sizeof *tots);
	*tot = zero;
	pp->i_cmds  = zero;
	pp->i_scpu  = zero;
	pp->i_ucpu  = zero;
	pp->i_uio   = zero;
	pp->i_uconn = zero;
	pp->i_rread = zero;
	pp->i_rpun  = zero;
	pp->i_rprt  = zero;
	pp->i_ublk  = zero;

	for (i=ifr; i<=ito; i++) {
		tp->t_cmds  += tots[index[i]].t_cmds;
		tp->t_scpu  += tots[index[i]].t_scpu;
		tp->t_ucpu  += tots[index[i]].t_ucpu;
		tp->t_uio   += tots[index[i]].t_uio ;
		tp->t_uconn += tots[index[i]].t_uconn;
		tp->t_rread += tots[index[i]].t_rread;
		tp->t_rpun  += tots[index[i]].t_rpun;
		tp->t_rprt  += tots[index[i]].t_rprt;
		tp->t_ublk  += tots[index[i]].t_ublk;
		pp->i_cmds  += perc[i].i_cmds ;
		pp->i_scpu  += perc[i].i_scpu ;
		pp->i_ucpu  += perc[i].i_ucpu ;
		pp->i_uio   += perc[i].i_uio  ;
		pp->i_uconn += perc[i].i_uconn;
		pp->i_rread += perc[i].i_rread;
		pp->i_rpun  += perc[i].i_rpun ;
		pp->i_rprt  += perc[i].i_rprt ;
		pp->i_ublk  += perc[i].i_ublk ;
		*tot        += overall[i];
	}
}


/*
* puthead: write out the report heading
*/
puthead(sdate, dept, page) char *sdate, *dept;
{
	printf("\f%20s%37s%-60s%4s%4d\n\n", sdate, "",
		"AU Resource Usage", "Page", page);
	printf("Department %s\n\n", dept);
	printf("%82s%s\r", "", "Unit Record I/O");
	printf("%82s%s\n", "", "_______________");
	printf("%19s%-15s%-13s%-13s%-19s%-26s%-16s%s\r",
		"", "Connect(hrs)", "#Commands", "Cpu(secs)", "Read/Writes",
		"Read  Punched  Printed", "Disk Blocks", "Overall%");
	printf("%19s%-15s%-13s%-13s%-19s%-26s%-16s%s\n",
		"", "____________", "_________", "_________", "___________",
		"______________________", "___________", "________");
	return(7);
}


/*
* puttot: output report total line
*/
puttot(tp, line, n) struct tots *tp;
{
	char *scsec();

	if (n >= 0)
		printf("\n%19s", tp->t_name); else
		printf("\n%19s", "        DEPT TOTALS");
        printf("%12s%12d%13.02f%15d%12d%9d%9d%15d",
	       scsec(tp->t_uconn, 12), tp->t_cmds,
	       (tp->t_scpu + tp->t_ucpu)/(float)1000, tp->t_uio, tp->t_rread,
	       tp->t_rpun, tp->t_rprt, tp->t_ublk);
	if (n >= 0)
		printf("%12.02f%%\n", overall[n]); else
		printf("\n");
	return(line+2);
}


/*
* putperc: output report total percent line
*/
putperc(pp, tot, line) struct rate *pp; float tot;
{
	float  pcpu;

	pcpu = pp->i_scpu + pp->i_ucpu / (float)2;
	printf("\n%19s%11.02f%%%11.02f%%%12.02f%%%14.02f%%%11.02f%%%8.02f%%%8.02f%%%14.02f%%%12.02f%%\n",
                "% OVER ALL AU",
		pp->i_uconn, pp->i_cmds, pcpu, pp->i_uio, pp->i_rread,
		pp->i_rpun, pp->i_rprt, pp->i_ublk, tot);

	return(line+2);
}


/*
* scsec: scales the given integer containing a second count into
*        hours, minutes and seconds and returns a pointer to a string.
*/
char *scsec(n, len)  int n, len;
{
	int  i, j, k;
	char ws[12];
	char *s;
	char *strcat();

	i = n/60;
	k = i/60;
	i = i%60;
	j = (int) n%60;
	sprintf(ws, "%d:%2d:%2d", k, i, j);

	s = ws;
	while(*++s)
		if (*s == ' ') *s = '0';
	i = strlen(ws);
	for (j=0; i<len; j++) {
		ss[j] = ' ';
		i++;
	}
	ss[j] = NULL;
	return(strcat(ss, ws));
}


/*
* zeroout: fills the buffer pointed to with zeros
*/
zeroout(s, len) int *s, len;
{
	register i;

	for (i=0; i<(len/BYTEPWD); i++)
		*s++ = 0;
}


int  kmr[13] = {
	0, 31, 59, 90, 120, 151, 181, 212, 243, 273, 304, 334, 365,
};
int  kml[13] = {
	0, 31, 60, 91, 121, 152, 182, 213, 244, 274, 305, 335, 366,
};


/*
* julian: converts the given gregorian date into julian and returns
*         the date as an integer.
*/
julian(gdate) char *gdate;
{
	int  gmonth, gday, gyear, days;
	char *s;

	s = gdate;
	gmonth = atoi(s) - 1;
	while (*s++ != '/') ;
	gday   = atoi(s);
	while (*s++ != '/') ;
	gyear  = atoi(s);

	days = (gyear%4)==0 ? kml[gmonth] : kmr[gmonth];

	return(gyear*1000 + days + gday);
}


/*
* greg: converts the given integer julian date to a gregorian
*       character date and returns a pointer to the string.
*/
char *greg(jdate)
{
	static char gdate[9];
	int  month, day, year, dayno, *kmp, i;

	year  = jdate/1000;
	dayno = jdate%1000;

	kmp = (year%4)==0 ? kml : kmr;

	for (i=0; i<12; i++) {
		if (dayno > kmp[i] && dayno <= kmp[i+1]) {
			month = i+1;
			day = dayno - kmp[i];
			break;
		}
	}
	sprintf(gdate, "%d/%d/%d", month, day, year);
	return(gdate);
}


/*
* any: scans the given string for the given character and returns 1 if
*      it is found, 0 otherwise.
*/
any(s, c)
char *s, c;
{

	while(*s)
		if(*s++ == c)
			return(1);
	return(0);
}
