/*
 * p/n:
 * Program name:           Dump Formatter system structure routines
 * Author:                 Alan Ballard
 * Backup:
 *
 * Description:
 *  This component contains the routines and table used for finding
 *  particular structures to print.
 */

#include "/usr/src/sys/h/param.h"
#include "/usr/src/sys/h/proc.h"
#include "/usr/src/sys/h/tty.h"
#include "/usr/src/sys/h/dir.h"
#include "/usr/src/sys/h/user.h"
#include "/usr/src/sys/h/coretab.h"
#include "/usr/src/sys/h/inode.h"
#include "/usr/src/sys/h/io.h"
#include "/usr/src/sys/h/ios.h"
#include "/usr/src/sys/h/text.h"
#include "/usr/src/sys/h/filsys.h"
#include "/usr/src/sys/h/ino.h"
#include "/usr/src/sys/h/buf.h"
#include "/usr/src/sys/h/ioconf.h"
#include "/usr/src/sys/h/dasd.h"
#include "/usr/src/sys/h/tube.h"
#include "/usr/src/sys/h/file.h"
#include "/usr/src/sys/h/mount.h"
#include "dump.h"
#include "tnode.h"
#include "y.tab.h"

/*
 *  FETCH is used to interface to getbytes for a word, halfword, etc.
 */
#define FETCH(loc) getbytes(&(loc), sizeof(loc))

/*
 *  define various routines which go in the structure table.
 */
int     prtactive();
int     prtbuf();
int     prtchan();
int     prtcrtab();
int     prtcu();
int     prtdasd();
int     prtfile();
int     prtinode();
int     prtioq();

int     prtmount();
int     prtproc();
int     prtstack();
int     prttext();
int     prttty();
int     prttube();
int     prtunit();
int     prtuser();
int     prtcommand();


int     filtbuf();
int     filtdasd();
int     filtfile();
int     filtinode();
int     filtmount();
int     filtproc();
int     filttext();
int     filttube();

/*
 *  The following table defines some properties of the various tables.
 *  It is used to avoid some long switch statements.
 */
struct structtable {
        char    *strucname;      /* the name of the structure */
        char    *symbol;         /* external symbol where it is rooted */
        int     elemsize;        /* size of each array element */
        int     numelems;        /* number of elements */
        int     (*prtrtn)();     /* routine to print it */
        int     (*filter)();     /* routine to filter during table scan */
        int     indirect;        /* indirect referencing required */

}      structtable[] = {

"buf",     "buf",   sizeof(struct buf),    NBUF,   prtbuf,   filtbuf,  0,
"chan",    "chans", sizeof(struct chan *), NCHANS, prtchan,  NULL,     1,
"coretab", NULL,    0,                     0,      prtcrtab, NULL,     0,
"cu",      NULL,    0,                     0,      prtcu,    NULL,     0,
"dasd",    "dasds", sizeof(struct dasd),   NDASD,  prtdasd,  filtdasd, 0,
"file",    "file",  sizeof(struct file),   NFILE,  prtfile,  filtfile, 0,
"inode",   "inode", sizeof(struct inode),  NINODE, prtinode, filtinode,0,
"ioq",     NULL,    0,                     0,      prtioq,   NULL,     0,
"mount",   "mount", sizeof(struct mount),  NMOUNT, prtmount, filtmount,0,
"proc",    "proc",  sizeof(struct proc),   NPROC,  prtproc,  filtproc, 0,
"text",    "text",  sizeof(struct text),   NTEXT,  prttext,  filttext, 0,
"tty",     NULL,    0,                     0,      prttty,   NULL,     0,
"tube",    "tptrs", sizeof(struct tube *), NTUBES, prttube,  NULL,     1,
"unit",    NULL,    0,                     0,      prtunit,  NULL,     0,
"user",    NULL,    0,                     0,      prtuser,  NULL,     0,
"command", NULL,    0,                     0,      prtcommand, NULL,   0,
"activeioq", NULL,  0,                     0,      prtactive,  NULL,   0,
"stack",   NULL,    0,                     0,      prtstack,   NULL,   0
};

/*
 *  prt just obtains the address of the printing routine from the
 *  structure table and calls it.
 */
prt(addr,code)
int     addr;         /* address to be formatted */
enum structcode code; /* code for what type of address */
{
        if (code < nonecode)
                (*(structtable[code].prtrtn))(addr);
        else
                error(0,"invalid structure code");
        return(0);
}

/*
 * findstruct is used for processing both tablename[int] and
 * unqualified tablename forms, since they have so much in common.
 *
 * For root structures, it will tour (or index into) the root
 * structure.  For others, it attempts (via recursive calls) to
 * tour higher-level structures to find something useful.
 */
findstruct(addr, code, table, index, expr, savflag)
int     addr;           /* address of a structure */
enum structcode code;   /* code for what type of structure we have */
enum structcode table;  /* code for structure we want */
int     index;          /* index for element wanted (-1 for all) */
struct tnode    *expr;  /* expression for what to do */
int     savflag;        /* whether to free expression nodes */
{
        struct tnode    *newgoal;
        register newaddr;
        register loop;

        if (table == expr->t_val)
            newgoal = expr->t_right;
        else
            newgoal = expr;

        if (code == nonecode) {
            struct structtable  *row;
            int     tablestart;

            row = structtable + table;

            /*
             * Don't have anything yet.  If this is a root structure,
             * set up symbols for touring/indexing.
             */
            if (row->symbol != NULL) {
                /*
                 *  Look up the symbol, and either tour the
                 *  structure, or access the required element.
                 */
                tablestart = symlook(row->symbol);
                if (index < 0)
                    for (loop = 0; loop < row ->numelems; loop ++) {
                        newaddr = tablestart + loop*(row->elemsize);
                        if (row->indirect)
                            newaddr = FETCH(*(int *)newaddr);
                        if (row->filter != NULL)
                            newaddr = (*(row->filter))(newaddr);
                        if (table == chancode)
                            io_chan = loop;       /* save for formatter */
                        printformat(newaddr, table, newgoal,savflag);

                    }
                else if (index < row->numelems) {
                    newaddr = tablestart + index*(row->elemsize);
                    if (row->indirect)
                        newaddr = FETCH(*(int *)newaddr);
                    if (table == chancode)
                        io_chan = index;      /* save for formatter */
                    printformat(newaddr, table, newgoal, savflag);
                } else
                    error(0, "invalid subscript for table");
                return(0);
            }
        }

        /*
         *  Have to find the required structure type from what we've been
         *  passed.  May mean a recursive call to find some higher
         *  level ones.
         */
        switch (table)  {
        case bufcode:
                 if (code == dasdcode) {
                         newaddr = FETCH(addr->d_devtab);
                         printformat(newaddr, bufcode, newgoal, savflag);
                 } else
                         error(0, "can't find buf");
                 break;

        case coretabcode:
                 if (newgoal != NULL || code != nonecode) {
                         error(0, "invalid specifications");
                         return(0);
                 } else
                         printformat(-1, coretabcode ,newgoal,savflag);
                 break;

        case inodecode:
                /*
                 *  can ask for inode from file, mount or text...
                 */
                switch (code) {
                case filecode:
                        newaddr = FETCH(addr->f_inode);
                        break;
                case mountcode:
                        newaddr = FETCH(addr->m_inodp);
                        break;
                case textcode:
                        newaddr = FETCH(addr->x_iptr);
                        break;
                default:
                        error(0, "can't find inode");
                        return(0);
                }
                printformat(newaddr, inodecode, newgoal, savflag);
                break;

        case usercode:
                if (code == proccode) {
                        newaddr = FETCH(addr->p_addr);
                        printformat(newaddr, usercode, newgoal, savflag);
                } else
                        /* Try to get some procs... */
                        findstruct(addr, code, proccode, -1, expr, savflag);
                break;

        case textcode:
                if (code == proccode) {
                        newaddr = FETCH(addr->p_textp);
                        printformat(newaddr, textcode, newgoal, savflag);
                } else
                        /* Try to get some procs... */
                        findstruct(addr, code, proccode, -1, expr, savflag);
                break;

        case ttycode:
                if (code == usercode) {
                        newaddr = FETCH(addr->u_ttyp);
                        printformat(newaddr, ttycode, newgoal, savflag);
                } else
                        /* Try to get some users... */
                        findstruct(addr, code, usercode, -1, expr, savflag);
                break;

        case commandcode:
                /*
                 *  "command" wants a proc table as its address
                 */
                if (code == proccode) {
                        printformat(addr, commandcode, newgoal, savflag);
                } else
                        /* Try to get some procs... */
                        findstruct(addr, code, proccode, -1, expr, savflag);
                break;

        case stackcode:
                /*
                 * "stack" wants the address of a register save area from
                 * which to start.  We assume the initial regs are
                 * in u_rsave in the user area.  This starts at R2,
                 * so we have to cheat a little.
                 */
                if (code == usercode) {
                        int *raddr;
                        raddr = &(addr->u_rsav);
                        raddr -= 2;        /* back up two words */
                        printformat((int)raddr,stackcode, newgoal,
                           savflag);
                } else
                        findstruct(addr, code, usercode, -1, expr, savflag);
                break;

        case ioqcode:
                /*
                 *  IOQ may be obtained from CHAN,  CU, or UNIT
                 */
                switch (code)  {

                case chancode:
                        newaddr = FETCH(addr->ch_q);
                        break;

                case cucode:
                        newaddr = FETCH(addr->un_q);
                        newaddr = FETCH(addr->cu_q);
                        break;

                case unitcode:
                        newaddr = FETCH(addr->un_q);
                        break;

                default:
                        error(0,"invalid ioq request");
                        return(0);
                }

                printformat(newaddr, ioqcode, newgoal, savflag);
                break;

        case activecode:
                /*
                 *  active ioq comes from a unit
                 */
                if (code == unitcode) {
                        newaddr = FETCH(addr->un_actv);
                        printformat(newaddr, activecode, newgoal, savflag);
                } else
                        /* Try to get some units... */
                        findstruct(addr, code, unitcode, -1, expr, savflag);
                break;


        case cucode:
                if (code == chancode)
                    if (index != -1)
                        if (index >= 16)  /* no #define for this */
                            error(0, "invalid control unit number");
                        else {
                            newaddr = FETCH(addr->ch_cu[index]);
                            if (io_chan != -1)
                                    io_cu = (io_chan << 4) + index;
                            printformat(newaddr,cucode, newgoal, savflag);
                        }
                    else
                        for (loop = 0; loop < 16; loop ++) {
                            newaddr = FETCH(addr->ch_cu[loop]);
                            if (io_chan != -1)
                                    io_cu = (io_chan << 4) + loop;
                            printformat(newaddr,cucode, newgoal, savflag);
                        }
                else
                    /* Try to get some chans... */
                    findstruct(addr, code, chancode, -1, expr, savflag);
                break;

        case unitcode:
                if (code == cucode)
                    if (index != -1)
                        if (index >= 16)  /* no #define for this */
                            error(0, "invalid device index");
                        else {
                            newaddr = FETCH(addr->cu_un[index]);
                            if (io_cu != -1)
                                    io_unit = (io_cu << 4) + index;
                            printformat(newaddr, unitcode, newgoal, savflag);
                        }
                    else
                        for (loop = 0; loop < 16; loop ++) {
                            newaddr = FETCH(addr->cu_un[loop]);
                            if (io_cu != -1)
                                    io_unit = (io_cu << 4) + loop;
                            printformat(newaddr, unitcode, newgoal, savflag);
                        }
                else
                    /* Try to get some cu's... */
                    findstruct(addr, code, cucode, -1, expr, savflag);
                break;

        default:
                error(0, "unimplemented specifications");
        }
}


/*
 *  do_proc is called to process the various selectors that
 *  select process structures.
 */
do_proc(addr, code, expr, savflag)
int     addr;           /* address of a structure */
enum structcode code;   /* code for what type of structure we have */
struct tnode    *expr;  /* expression for what to do */
int     savflag;        /* whether to free expression nodes */
{
        register temp, want;

        if (code == proccode) {
                want = expr->t_left;

                switch (expr->t_val) {
                case UID:
                        temp = FETCH(addr->p_uid);
                        if (temp != want)
                                return(0);
                        break;

                case PID:
                        temp = FETCH(addr->p_pid);
                        if(temp != want)
                                return(0);
                        break;

                case PGROUP:
                        temp = FETCH(addr->p_pgrp);
                        if (temp != want)
                                return(0);
                        break;

                default:
                        error(0,"error in do_proc");
                        return(0);
                }
                printformat(addr, proccode, expr->t_right, savflag);
        } else
                findstruct(addr, code, proccode, -1, expr, savflag);
}

/*
 * do_dev is called to find device with a specific device address.
 */
do_dev(addr,code,expr,savflag)
int     addr;           /* address of a structure */
enum structcode code;   /* code for what type of structure we have */
struct tnode    *expr;  /* expression for what to do */
int     savflag;        /* whether to free expression nodes */
{
        struct chan    **chantabloc;
        struct chan    *chantab;
        struct cu      *cutab;
        struct unit    *unittab;
        if (code != nonecode) {
                error(0,"invalid specifications");
                return 0;
        }
        io_unit = expr->t_val;
        io_cu = io_unit >> 4;
        io_chan = io_cu >> 4;

        chantabloc = symlook("chans");
        chantabloc += io_chan;
        chantab =  FETCH(*chantabloc);
        cutab = FETCH(chantab->ch_cu[io_cu & 0x0f]);
        unittab = FETCH(cutab->cu_un[io_unit & 0x0f]);
        printformat(unittab, unitcode, expr->t_right, savflag);
}

/*
 *  Filtering routines used during scans of tables to suppress elements
 *  which aren't in use.
 *
 *  Each of these routines is called with an address and returns either
 *  that address or zero.
 */
filtbuf(addr)
int     addr;
{
        register temp;
        temp = FETCH(addr->b_flags);
        if (temp & B_BUSY)
                return (addr);
        return (0);
}

filtdasd(addr)
int     addr;
{
        register temp;
        temp = FETCH(addr->d_desc);
        if (temp)
                return (addr);
        return (0);
}

filtfile(addr)
int     addr;
{
        register temp;
        temp = FETCH(addr->f_count);
        if (temp > 0)
                return (addr);
        return (0);
}

filtinode(addr)
int     addr;
{
        register temp;
        temp = FETCH(addr->i_count);
        if (temp > 0)
                return (addr);
        return (0);
}

filtmount(addr)
int     addr;
{
        register temp;
        temp = FETCH(addr->m_inodp);
        if (temp != 0)
                return (addr);
        return (0);
}

filtproc(addr)
int     addr;
{
        register temp;
        temp = FETCH(addr->p_stat);
        if (temp != 0)
                return (addr);
        return (0);
}

filttext(addr)
int     addr;
{
        register temp;
        temp = FETCH(addr->x_count);
        if (temp != 0)
                return (addr);
        return (0);
}

